"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AllowanceTransfer = void 0;
const tslib_1 = require("tslib");
const tiny_invariant_1 = tslib_1.__importDefault(require("tiny-invariant"));
const hash_1 = require("@ethersproject/hash");
const constants_1 = require("./constants");
const domain_1 = require("./domain");
const PERMIT_DETAILS = [
    { name: 'token', type: 'address' },
    { name: 'amount', type: 'uint160' },
    { name: 'expiration', type: 'uint48' },
    { name: 'nonce', type: 'uint48' },
];
const PERMIT_TYPES = {
    PermitSingle: [
        { name: 'details', type: 'PermitDetails' },
        { name: 'spender', type: 'address' },
        { name: 'sigDeadline', type: 'uint256' },
    ],
    PermitDetails: PERMIT_DETAILS,
};
const PERMIT_BATCH_TYPES = {
    PermitBatch: [
        { name: 'details', type: 'PermitDetails[]' },
        { name: 'spender', type: 'address' },
        { name: 'sigDeadline', type: 'uint256' },
    ],
    PermitDetails: PERMIT_DETAILS,
};
function isPermit(permit) {
    return !Array.isArray(permit.details);
}
class AllowanceTransfer {
    /**
     * Cannot be constructed.
     */
    constructor() { }
    // return the data to be sent in a eth_signTypedData RPC call
    // for signing the given permit data
    static getPermitData(permit, permit2Address, chainId) {
        (0, tiny_invariant_1.default)(constants_1.MaxSigDeadline.gte(permit.sigDeadline), 'SIG_DEADLINE_OUT_OF_RANGE');
        const domain = (0, domain_1.permit2Domain)(permit2Address, chainId);
        if (isPermit(permit)) {
            validatePermitDetails(permit.details);
            return {
                domain,
                types: PERMIT_TYPES,
                values: permit,
            };
        }
        else {
            permit.details.forEach(validatePermitDetails);
            return {
                domain,
                types: PERMIT_BATCH_TYPES,
                values: permit,
            };
        }
    }
    static hash(permit, permit2Address, chainId) {
        const { domain, types, values } = AllowanceTransfer.getPermitData(permit, permit2Address, chainId);
        return hash_1._TypedDataEncoder.hash(domain, types, values);
    }
}
exports.AllowanceTransfer = AllowanceTransfer;
function validatePermitDetails(details) {
    (0, tiny_invariant_1.default)(constants_1.MaxOrderedNonce.gte(details.nonce), 'NONCE_OUT_OF_RANGE');
    (0, tiny_invariant_1.default)(constants_1.MaxAllowanceTransferAmount.gte(details.amount), 'AMOUNT_OUT_OF_RANGE');
    (0, tiny_invariant_1.default)(constants_1.MaxAllowanceExpiration.gte(details.expiration), 'EXPIRATION_OUT_OF_RANGE');
}
//# sourceMappingURL=allowanceTransfer.js.map